package furny.swing.admin.viewer;

import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.util.concurrent.Callable;

import com.jme3.font.BitmapFont;
import com.jme3.font.BitmapFont.Align;
import com.jme3.font.BitmapText;
import com.jme3.font.Rectangle;
import com.jme3.input.ChaseCamera;
import com.jme3.input.InputManager;
import com.jme3.input.KeyInput;
import com.jme3.input.controls.ActionListener;
import com.jme3.input.controls.KeyTrigger;
import com.jme3.input.controls.MouseButtonTrigger;
import com.jme3.material.Material;
import com.jme3.math.ColorRGBA;
import com.jme3.math.FastMath;
import com.jme3.math.Vector3f;
import com.jme3.renderer.Camera;
import com.jme3.scene.Node;
import com.jme3.scene.Spatial;
import com.jme3.scene.Spatial.CullHint;

import furny.entities.Furniture;
import furny.furndb.FurnDBManager;
import furny.furndb.importer.DBModelSource;
import furny.furndb.importer.IModelSource;
import furny.jme.FurnyApplication;
import furny.jme.appstate.SimpleAppState;
import furny.jme.node.Arrow;
import furny.jme.node.Grid;
import furny.util.CameraUtils;
import furny.util.LightingUtil;
import furny.util.ModelUtil;

/**
 * App state for the model viewer.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public class ModelViewerState extends SimpleAppState implements IModelViewer {
  private final IModelSource modelSource;

  private ChaseCam2 chaseCam;

  private Node camTarget;
  private Node modelNode;

  private Node axisNode;
  private Node arrowNode;

  private Node scaleGrid, userScaleGrid;

  private BitmapText infoText, infoText2, infoText3;
  private BitmapFont consoleFont;

  private Material redMaterial;
  private Material greenMaterial;
  private Material blueMaterial;

  private Furniture furniture;

  private boolean rotating;
  private boolean manual;
  private boolean showGrid;

  private boolean showScaleGrid;
  private boolean showUserScaleGrid;
  private boolean showDirectionArrow = true;

  private boolean hasModel;

  private final NumberFormat floatFormat = new DecimalFormat("0.00");

  private final FurnyApplication mainApp;

  /**
   * Instantiates a new model viewer app state.
   * 
   * @param modelSource
   *          the model source
   * @param app
   *          the app
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public ModelViewerState(final IModelSource modelSource,
      final FurnyApplication app) {
    this.modelSource = modelSource;
    this.mainApp = app;
  }

  @Override
  public void simpleInitAppState() {
    camTarget = new Node("Camera Target");
    modelNode = new Node("Model Node");

    flyCam.setEnabled(false);

    chaseCam = new ChaseCam2(cam, camTarget, inputManager);
    chaseCam.setMaxDistance(20f);
    chaseCam.setMinDistance(0.01f);
    chaseCam.setMinVerticalRotation(-FastMath.PI / 2f);
    chaseCam.setInvertVerticalAxis(false);
    chaseCam.setZoomSensitivity(20f);
    chaseCam.setZoomSpeed(0.02f);
    chaseCam.setRotationSensitivity(20f);
    chaseCam.setDragToRotate(true);
    chaseCam.setSmoothMotion(false);
    viewPort.setBackgroundColor(ColorRGBA.White);

    inputManager.addMapping("manual", new MouseButtonTrigger(0),
        new MouseButtonTrigger(1), new MouseButtonTrigger(2));
    inputManager.addListener(new ActionListener() {
      @Override
      public void onAction(final String name, final boolean isPressed,
          final float tpf) {
        manual = isPressed;
      }
    }, "manual");

    inputManager.addMapping("nextModel", new KeyTrigger(KeyInput.KEY_RIGHT));
    inputManager.addMapping("previousModel", new KeyTrigger(KeyInput.KEY_LEFT));
    inputManager.addListener(new ActionListener() {
      @Override
      public void onAction(final String name, final boolean isPressed,
          final float tpf) {
        if (isPressed) {
          if ("nextModel".equals(name)) {
            loadModel(1);
          } else if ("previousModel".equals(name)) {
            loadModel(-1);
          }
        }
      }
    }, "nextModel", "previousModel");

    rootNode.attachChild(camTarget);
    rootNode.attachChild(modelNode);

    greenMaterial = new Material(assetManager,
        "Common/MatDefs/Misc/Unshaded.j3md");
    greenMaterial.getAdditionalRenderState().setDepthTest(false);
    greenMaterial.setColor("Color", ColorRGBA.Green);

    redMaterial = new Material(assetManager,
        "Common/MatDefs/Misc/Unshaded.j3md");
    redMaterial.getAdditionalRenderState().setDepthTest(false);
    redMaterial.setColor("Color", ColorRGBA.Red);

    blueMaterial = new Material(assetManager,
        "Common/MatDefs/Misc/Unshaded.j3md");
    blueMaterial.getAdditionalRenderState().setDepthTest(false);
    blueMaterial.setColor("Color", ColorRGBA.Blue);

    // axis
    axisNode = new Grid(new Vector3f(1f, 1f, 1f), 0.25f, 1f, assetManager);
    axisNode.setMaterial(redMaterial);
    rootNode.attachChild(axisNode);

    arrowNode = new Arrow(Vector3f.ZERO, Vector3f.UNIT_Z, 0.06f, 3f,
        assetManager);
    arrowNode.setMaterial(blueMaterial);
    rootNode.attachChild(arrowNode);

    setShowGrid(false);

    LightingUtil.createDefaultLights(rootNode);

    infoText = new BitmapText(guiFont, false);

    infoText.setText("");
    infoText.setColor(ColorRGBA.Black);

    consoleFont = assetManager.loadFont("Interface/Fonts/Console.fnt");

    infoText2 = new BitmapText(consoleFont, false);

    infoText2.setText("");
    infoText2.setColor(ColorRGBA.Black);

    infoText3 = new BitmapText(consoleFont, false);

    infoText3.setText("");
    infoText3.setColor(ColorRGBA.Black);
    arrangeText(mainApp.getAppSettings().getWidth(), mainApp.getAppSettings()
        .getHeight());

    guiNode.attachChild(infoText);
    guiNode.attachChild(infoText2);
    guiNode.attachChild(infoText3);

    fpsText.setColor(ColorRGBA.Black);

    setShowStatistics(isShowStatistics());

    for (final Spatial sp : statsView.getChildren()) {
      if (sp instanceof BitmapText) {
        ((BitmapText) sp).setColor(ColorRGBA.Black);
      }
    }

    if (!hasModel) {
      loadModel(0);
    }
  }

  /*
   * (non-Javadoc)
   * 
   * @see furny.furndb.admin.viewer.IModelViewer#setShowGrid(boolean)
   * 
   * @since 25.06.2011
   * 
   * @author stephan
   */
  @Override
  public void setShowGrid(final boolean show) {
    this.showGrid = show;
    mainApp.enqueue(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        axisNode.setCullHint(show ? CullHint.Never : CullHint.Always);
        return null;
      }
    });
  }

  /*
   * (non-Javadoc)
   * 
   * @see furny.furndb.admin.viewer.IModelViewer#isShowGrid()
   * 
   * @since 25.06.2011
   * 
   * @author stephan
   */
  @Override
  public boolean isShowGrid() {
    return showGrid;
  }

  @Override
  public boolean isShowDirection() {
    return showDirectionArrow;
  }

  @Override
  public void setShowDirection(final boolean showDirectionArrow) {
    this.showDirectionArrow = showDirectionArrow;

    mainApp.enqueue(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        arrowNode.setCullHint(showDirectionArrow ? CullHint.Never
            : CullHint.Always);
        return null;
      }
    });
  }

  /*
   * (non-Javadoc)
   * 
   * @see furny.furndb.admin.viewer.IModelViewer#isRotating()
   * 
   * @since 25.06.2011
   * 
   * @author stephan
   */
  @Override
  public boolean isRotating() {
    return rotating;
  }

  /*
   * (non-Javadoc)
   * 
   * @see furny.furndb.admin.viewer.IModelViewer#setRotating(boolean)
   * 
   * @since 25.06.2011
   * 
   * @author stephan
   */
  @Override
  public void setRotating(final boolean rotating) {
    this.rotating = rotating;
  }

  /**
   * Checks if the context is closed.
   * 
   * @return true, if is closed
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public boolean isClosed() {
    return !mainApp.getContext().isCreated();
  }

  /**
   * Load a model.
   * 
   * @param offset
   *          the index offset of the model to load. (1 for next, -1 for
   *          previous)
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private void loadModel(final int offset) {
    final Furniture furn = modelSource.next(offset);
    // model = GeometryBatchFactory.optimize(model);

    if (furn != null) {
      setFurniture(furn);
    }
  }

  /*
   * (non-Javadoc)
   * 
   * @see
   * furny.furndb.admin.viewer.IModelViewer#setFurniture(furny.entities.Furniture
   * )
   * 
   * @since 25.06.2011
   * 
   * @author stephan
   */
  @Override
  public void setFurniture(final Furniture furn) {
    this.furniture = furn;
    hasModel = true;
    mainApp.enqueue(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        modelNode.detachAllChildren();

        if (furn != null) {
          modelNode.attachChild(furn.getModel());

          // add the scale Grid
          final Vector3f extents = ModelUtil.getExtents(furn.getModel());
          final Vector3f halfExtents = extents.divide(2f);
          final Vector3f center = furn.getModel().getWorldBound().getCenter();
          scaleGrid = new Grid(extents, 0f, 0f, 2f, assetManager);

          scaleGrid.setMaterial(redMaterial);

          modelNode.attachChild(scaleGrid);
          scaleGrid.setLocalTranslation(center.subtract(halfExtents));

          // add the user scale grid
          final Vector3f userExtents = furn.getMetaData().getDimension();

          userScaleGrid = new Grid(userExtents, 0f, 0f, 2f, assetManager);
          userScaleGrid.setMaterial(greenMaterial);

          modelNode.attachChild(userScaleGrid);
          userScaleGrid.setLocalTranslation(center.subtract(userExtents
              .divide(2f)));
          //

          infoText.setText(furn.getName() + " ("
              + furn.getModel().getTriangleCount() + " Triangles)");

          infoText2.setText("scale: " + floatFormat.format(extents.x) + ", "
              + floatFormat.format(extents.y) + ", "
              + floatFormat.format(extents.z));
          infoText3.setText("user scale: " + floatFormat.format(userExtents.x)
              + ", " + floatFormat.format(userExtents.y) + ", "
              + floatFormat.format(userExtents.z));

          setShowScaleGrid(showScaleGrid);
          setShowUserScaleGrid(showUserScaleGrid);

          CameraUtils.adjustChaseCam(furn.getModel(), chaseCam, cam, camTarget);

          modelSource.seek(furn);

        } else {
          infoText.setText("");

          infoText2.setText("");
          infoText3.setText("");
        }

        rootNode.updateGeometricState();
        guiNode.updateGeometricState();

        return null;
      }
    });
  }

  @Override
  public boolean isShowScaleGrid() {
    return showScaleGrid;
  }

  @Override
  public void setShowScaleGrid(final boolean showScaleGrid) {
    this.showScaleGrid = showScaleGrid;

    mainApp.enqueue(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        scaleGrid.setCullHint(showScaleGrid ? CullHint.Never : CullHint.Always);
        return null;
      }
    });
  }

  @Override
  public boolean isShowUserScaleGrid() {
    return showUserScaleGrid;
  }

  @Override
  public void setShowUserScaleGrid(final boolean showUserScaleGrid) {
    this.showUserScaleGrid = showUserScaleGrid;

    mainApp.enqueue(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        userScaleGrid.setCullHint(showUserScaleGrid ? CullHint.Never
            : CullHint.Always);
        return null;
      }
    });
  }

  @Override
  public void simpleUpdate(final float tpf) {
    if (!manual && rotating) {
      chaseCam.setDefaultHorizontalRotation(chaseCam.getHorizontalRotation()
          + .5f * tpf);
    }
  }

  @Override
  public void centerFurniture(final boolean save) {
    if (furniture != null) {
      final Node n = furniture.getModel();
      ModelUtil.center(n);

      CameraUtils.adjustChaseCam(n, chaseCam, cam, camTarget);

      if (save) {
        // re-set model to update the binary data
        furniture.setModel(n);

        FurnDBManager.getInstance().saveFurniture(furniture, true);
      }
    }
  }

  @Override
  public void rotateFurniture(final float rotation, final boolean save) {
    if (furniture != null) {
      final Node n = furniture.getModel();
      n.rotate(0, rotation, 0);

      CameraUtils.adjustChaseCam(n, chaseCam, cam, camTarget);

      if (save) {
        furniture.setModel(n);

        FurnDBManager.getInstance().saveFurniture(furniture, true);
      }
    }
  }

  @Override
  public void arrangeText(final int width, final int height) {
    mainApp.enqueue(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        if (infoText != null && infoText2 != null && infoText3 != null) {
          infoText.setLocalTranslation(0f, height, 0f);
          infoText.setBox(new Rectangle(0f, 0f, width, infoText.getLineHeight()));
          infoText.setAlignment(Align.Center);

          infoText2.setLocalTranslation(0f, height - infoText.getLineHeight(),
              0f);
          infoText2.setBox(new Rectangle(0f, 0f, width, infoText2
              .getLineHeight()));
          infoText2.setAlignment(Align.Center);

          infoText3.setLocalTranslation(0f, height - infoText.getLineHeight()
              - infoText2.getLineHeight(), 0f);
          infoText3.setBox(new Rectangle(0f, 0f, width, infoText3
              .getLineHeight()));
          infoText3.setAlignment(Align.Center);

          guiNode.updateGeometricState();
        }

        return null;
      }
    });
  }

  /**
   * Extended {@link ChaseCamera} that allows to set the zoom speed.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  private final class ChaseCam2 extends ChaseCamera {

    /**
     * Instantiates a new chase cam 2.
     * 
     * @param cam
     *          the cam
     * @param target
     *          the target
     * @param inputManager
     *          the input manager
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public ChaseCam2(final Camera cam, final Node target,
        final InputManager inputManager) {
      super(cam, target, inputManager);
    }

    /**
     * Sets the zoom speed.
     * 
     * @param zoomSpeed
     *          the new zoom speed
     * @since 12.08.2012
     * @author Stephan Dreyer
     */
    public void setZoomSpeed(final float zoomSpeed) {
      this.zoomSpeed = zoomSpeed;
    }
  }

  /**
   * Main method to test the class.
   * 
   * @param args
   *          No arguments required.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public static void main(final String[] args) {
    IModelSource source;
    // source = new FileModelSource("models/new/");
    source = new DBModelSource();

    final FurnyApplication app = new FurnyApplication();

    final ModelViewerState state = new ModelViewerState(source, app);

    app.enqueue(new Callable<Void>() {
      @Override
      public Void call() throws Exception {
        app.getStateManager().attach(state);
        return null;
      }
    });

    app.start();
  }
}
